/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence,  or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include "external.h"
#include "common.h"
#include "headers.h"
#include "url.h"
#include "quote.h"
#include <boost/process.hpp>

//extern "C" {
//  #include <curl/curl.h>
//}

using namespace boost::process;
using boost::lexical_cast;
using boost::bad_lexical_cast;

int call_curl (const ::std::string& cmdline, const int oops, bool pure_code, ::std::ostringstream& ss)
{   int code = 0;
    try
    {   ipstream pipe_stream;
        child ext (cmdline, std_out > pipe_stream);
        if (ext.valid () && pipe_stream)
        {   ::std::string line;
            bool first = true;
            while (pipe_stream && ::std::getline (pipe_stream, line))
            {   if (! line.empty () && pure_code)
                    code = lexical < int > :: cast (line, 3);
                else if (first && ! line.empty ())
                {   headers h (line, ss);
                    code = h.code ();
                    first = false; }
                if (code > 0) break; } }
        ext.wait (); }
    catch (...)
    {   if (context.tell (e_severe)) ss << "Cannot check external links. Please verify your installation of curl\n";
        code = oops;
        context.external (false); }
    return code; }

void test_hypertext (const url& u, ::std::ostringstream& ss)
{   ::std::string cmdline ("curl -o NUL --silent --head --write-out %{http_code} ");
    if (u.is_https () && ! context.revoke ()) cmdline += "--ssl-norevoke ";
    cmdline += u.original ();
    if (context.tell (e_debug)) ss << context.filename () <<  " : " <<  quote (cmdline)<< "\n";
    int code = call_curl (cmdline, 599, true, ss);
    if (code)
    {   context.code (code);
        if (context.tell (e_variable)) ss << "got " << ::boost::lexical_cast <::std::string> (code) << "\n"; } }

void test_connection (const url& u, ::std::ostringstream& ss)
{   if (u.is_usable ()) test_hypertext (u, ss);
    else if (context.tell (e_error)) ss << context.filename () << " unable to test " << u.original () << "\n"; }

bool external::verify (const url& u, ::std::ostringstream& ss)
{   if (! context.external ()) return true;
    if (u.empty ()) { context.code (400); return false; }
    auto e = url_.find (u.absolute ());
    context.repeated (e != url_.end ());
    if (context.repeated ()) context.code (e -> second);
    else
    {   test_connection (u, ss);
        url_.insert (value_t (u.absolute (), context.code ())); }
    if ((context.code () >= 400) && (context.code () < 500)) return false;
    if (! context.forwarded ()) return true;
    return ((context.code () != 301) && (context.code () != 308)); };

bool fetch_common (const url& u, const ::boost::filesystem::path& file, const char* curl, ::std::ostringstream& ss)
{   if (! u.is_usable ())
    {   if (context.tell (e_error)) ss << "Unable to open " << u.original () << " to find " WEBMENTION "\n";
        return false; }
    ::std::string cmdline (curl);
    cmdline += file.string ();
    cmdline += " ";
    if (u.is_https () && context.revoke ()) cmdline += "--ssl-norevoke ";
    cmdline += u.absolute ();
    if (context.tell (e_debug)) ss << WEBMENTION " http headers sought with: " << quote (cmdline) << "\n";
    int code = call_curl (cmdline, 599, false, ss);
    if (context.tell (e_variable)) ss << "got " << ::boost::lexical_cast <::std::string> (code) << "\n";
    return (code < 300); }

bool fetch_http (const url& u, const ::boost::filesystem::path& file, ::std::ostringstream& ss)
{   return fetch_common (u, file, "curl -I -o ", ss); }

bool fetch (const url& u, const ::boost::filesystem::path& file, ::std::ostringstream& ss)
{   return fetch_common (u, file, "curl -o ", ss); }

bool mention (const url& source, const url& target, const url& server, ::std::ostringstream& ss)
{   typedef ssc_map < ::std::string, bool > validity_t;
    static validity_t validity;
    if (! context.nochange ())
    {   auto x = validity.find (server.absolute ());
        if ((x != validity.end ())) if (! x -> second) return false; }
    ::std::string cmdline ("curl -i -d source=");
    cmdline += source.absolute (true);
    cmdline += " -d target=";
    cmdline += target.absolute ();
    cmdline += " ";
    if (server.is_https () && ! context.revoke ()) cmdline += "--ssl-norevoke ";
    cmdline += server.absolute ();
    if (context.tell (e_debug)) ss << quote (cmdline) << "\n";
    if (context.nochange ()) return true;
    int code = call_curl (cmdline, 599, false, ss);
    if (context.tell (e_variable)) ss << "got " << ::boost::lexical_cast <::std::string> (code) <<  "\n";
    validity.insert (validity_t::value_type (server.absolute (), code < 300));
    return (code < 300); }
