/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence,  or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#pragma once
#include "symbol.h"
#include "vocab.h"
#include "microformat_properties.h"
#include <memory>

// categories
constexpr int c_generic = 0;
constexpr int c_html3 = 0x00000001;
constexpr int c_microformat = 0x00000002;
constexpr int c_html4 = 0x00000004;
constexpr int c_xfn = 0x00000008;
constexpr int c_its = 0x00000010;
constexpr int c_grddl = 0x00000020;
constexpr int c_vcs = 0x00000040;
constexpr int c_atom = 0x00000080;  // RFC 4287
constexpr int c_amp_html = 0x00000100;
constexpr int c_safari = 0x00000200;
constexpr int c_indieauth = 0x00000400;
constexpr int c_dcterms = 0x00000800;
constexpr int c_lightbox = 0x00001000;
constexpr int c_openid = 0x00002000;
constexpr int c_hcal = 0x00004000;
constexpr int c_cc = 0x00008000;
constexpr int c_rejected = 0x10000000;
constexpr int c_dropped = 0x20000000;
constexpr int c_draft = 0x40000000;


// effects on link, a, area
typedef enum
{   ela_no, ela_ok, ela_external, ela_hyperlink, ela_popup, c_annotation, ela_external_contextual }
e_linkaarea;

class microformat_base
{
protected:
    bool declared_;
public:
    microformat_base () : declared_ (false) {}
    microformat_base (const microformat_base&) = default;
    microformat_base (microformat_base&&) = default;
    microformat_base& operator = (const microformat_base&) = default;
    microformat_base& operator = (microformat_base&&) = default;
    virtual ~microformat_base () = default;
    explicit microformat_base (bool b) : declared_ (b) {}

    virtual e_vocabulary vocab () const { return v_error; }
    virtual bool has () const { return false; }
    virtual void get (void** ) { }
    virtual void get (const void** ) const { }
    virtual bool is_known_name () { return false; }
    virtual bool declared () const { return declared_; }
    virtual void declare (bool b = true) { declared_ = b; }

    virtual bool is_relational () const { return false; }
    virtual void validate_element (const size_t ) const { }
    virtual void set_mf_value (const e_property , element& ) { }
    virtual ::std::string get_value (const e_property ) { return ::std::string (); }
    virtual bool has_prop (const e_property ) const { return false; }
    virtual void reset () { declared_ = false; }
    virtual void swap (microformat_base& mf) noexcept
    {   ::std::swap (declared_, mf.declared_); }
    virtual bool invalid () const { return true; }
    virtual bool has_url () const { return false; }
    virtual bool verify_url (const directory& ) { return true; }
    virtual bool empty () const { return true; }
    virtual ::std::string diagnose (const int ) const { return ::std::string (); }
    virtual ::std::string report (const int ) const { return ::std::string (); } };

typedef ::std::shared_ptr < microformat_base > microformat_base_ptr;

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > class microformat : public microformat_base
{   typedef ::std::tuple < PROPERTIES... > property_t;
    property_t p_;
public:
    microformat () = default;
    microformat (const microformat&) = default;
    microformat (microformat&&) = default;
    microformat& operator = (const microformat&) = default;
    microformat& operator = (microformat&&) = default;
    virtual ~microformat () = default;
    explicit microformat (bool b) : microformat_base (b) {}
    virtual void reset ();
    virtual void swap (microformat& mf) noexcept;
    constexpr static e_vocabulary whoami () { return VOCAB; }
    virtual e_vocabulary vocab () const { return VOCAB; }
    template < class PROPERTY > bool has () const
    {   return has_property < PROPERTY, PROPERTIES... > (); }
    template < class PROPERTY > void get (PROPERTY** p)
    {   get_property < PROPERTY, PROPERTIES... > (p_, p); }
    template < class PROPERTY > void get (const PROPERTY** p) const
    {   get_property < PROPERTY, PROPERTIES... > (p_, p); }
    static ::std::string name () { return vocab::name (VOCAB); }
    template < e_property PROP > bool is_known_name ()
    {   bool res = false;
        for_each_attribute (p_, [&](auto& p)
            {   if (! res) res = p.whoami () == PROP; } );
        return res; }
    virtual bool is_relational () const;
    virtual void validate_element (const size_t e) const;
    virtual void set_mf_value (const e_property pp, element& e);
    virtual ::std::string get_value (const e_property pp);
    virtual bool has_prop (const e_property p) const;
    virtual bool invalid () const;
    virtual bool has_url () const;
    virtual bool verify_url (const directory& d);
    virtual bool empty () const;
    virtual ::std::string diagnose (const int n) const;
    virtual ::std::string report (const int n) const; };


template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > bool microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: has_prop (const e_property p) const  // I can do better here
{   bool res = false;
    for_each_attribute (p_, [&](auto t)
        {   res = res || (t.whoami () == p); } );
    return res; }

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > void microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: reset ()
{   microformat_base::reset ();
    microformat mf;
    swap (mf); }

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > void microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: swap (microformat& mf) noexcept
{   ::std::swap (p_, mf.p_);
    microformat_base::swap (mf); }

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > bool microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: is_relational () const
{   bool res = false;
    for_each_attribute (p_, [&](auto& p)
        {   res = res || p.is_relational (); } );
    return res; }

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > void microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: validate_element (const size_t e) const
{   for_each_attribute (p_, [&](auto& p)
        {   p.validate_element (e); } ); }

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > void microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: set_mf_value (const e_property pp, element& e)
{   for_each_attribute (p_, [&](auto& p)
        {   if (p.whoami () == pp) p.set_mf_value (e); } ); }

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > ::std::string microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: get_value (const e_property pp)
{   ::std::string res;
    for_each_attribute (p_, [&](auto& p)
        {   if (p.whoami () == pp) res = p.get_value (); } );
    return res; }

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > bool microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: invalid () const
{   if (! declared_) return true;
    bool res = false;
    for_each_attribute (p_, [&](auto& p)
        {   res = res || p.invalid (); } );
    return res; }

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > bool microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: has_url () const
{   bool res = false;
    for_each_attribute (p_, [&](auto& p)
        {   res = res || p.is_url (); } );
    return res; }

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > bool microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: verify_url (const directory& d)
{   bool res = true;
    for_each_attribute (p_, [&](auto& p)
        {   res = res && p.verify_url (d); } );
    return res; }

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > bool microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: empty () const
{   if (! declared_) return true;
    bool res = true;
    for_each_attribute (p_, [&](auto& p)
        {   res = res && p.empty (); } );
    return res; }

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > ::std::string microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: diagnose (const int n) const
{  ::std::string res;
    for_each_attribute (p_, [&](auto& p)
    {   if (! p.unknown ()) res += p.diagnose (); } );
    if (res.empty ()) return res;
    ::std::string doris (fyi (n));
    doris += name ();
    if (context.tell (e_detail))
    {   doris += " (";
        doris += ::boost::lexical_cast < ::std::string > (VOCAB);
        doris += ")"; }
    doris += "\n";
    doris += res;
    return doris; }

template < e_vocabulary VOCAB, int CATEGORY, e_linkaarea LINK, e_linkaarea A_AREA, class... PROPERTIES > ::std::string microformat < VOCAB, CATEGORY, LINK, A_AREA, PROPERTIES... > :: report (const int n) const
{   if (! context.tell (e_comment)) return ::std::string ();
    ::std::string res (fyi (n));
    bool xtra = context.tell (e_detail);
    if constexpr (VOCAB < first_rel_vocab)
    {   res += name ();
        if (xtra) res += " "; }
    if (xtra)
    {   res += "(";
        res += vocab::name (VOCAB);
        res += ",";
        res += ::boost::lexical_cast < ::std::string > (VOCAB);
        res += ")"; }
    bool first = true;
    for_each_attribute (p_, [&](auto& p)
    {   if (! p.unknown () || context.tell (e_splurge))
        {   if (xtra)
                if (first) { res += ": "; first = false; }
                else res += "; ";
            if (! is_relational ())
            {   res += p.name ();
                res += "="; }
            res += p.report (); } } );
    res += "\n";
    return res; }
