/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence,  or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#pragma once

#include "type_core.h"
#include "symbol.h"
#include "version.h"
#include "vocab.h"
#include "quote.h"

template < e_type TYPE > class type_master : public string_value < TYPE > { };

template < > struct type_master < t_existential > : public type_base < t_existential >
{   static bool is_existential () { return true; }
    void set_value (const ::std::string& s) // sanity test only
    {   if (s.empty ()) type_base < t_existential > :: status (s_good);
        else type_base < t_existential > :: status (s_invalid); }
    ::std::string diagnose () const
    {   if (! type_base < t_existential > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        return "may not be given a value"; } };

template < > struct type_master < t_date > : public string_value < t_date >
{   void set_value (const ::std::string& s) // sanity test only
    {   ::std::string val (trim_the_lot_off (s));
        string_value < t_date > :: set_value (val);
        if (val.find_first_not_of (DDD ":+ZTzt") != val.npos)
            string_value < t_date > :: status (s_invalid); }
    ::std::string diagnose () const
    {   if (! string_value < t_date > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        ::std::string s (quote (get_value ()));
        s += " is an invalid datetime (see RFC-6350)";
        return s; } };

template < > struct type_master < t_duration > : public string_value < t_duration >
{   void set_value (const ::std::string& s) // sanity tests only
    {   ::std::string val (trim_the_lot_off (s));
        string_value < t_duration > :: set_value (val);
        if (val.empty ())
            string_value < t_duration > :: status (s_invalid);
        else if (val [0] != 'P' && val [0] != 'p')
            string_value < t_duration > :: status (s_invalid);
        else if (val.find_first_not_of (DDD ":+PYMWDTHSRpymdthsr/") != val.npos) // https://en.wikipedia.org/wiki/ISO_8601
            string_value < t_duration > :: status (s_invalid); }
    ::std::string diagnose () const
    {   if (! string_value < t_duration > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        ::std::string s (quote (get_value ()));
        s += " is an invalid duration (see RFC-6350)";
        return s; } };


template < > struct type_master < t_target > : public string_value < t_target >
{   void set_value (const ::std::string& s)
    {   ::std::string val (trim_the_lot_off (s));
        string_value < t_target > :: set_value (val);
        if (val.empty ())
            string_value < t_target > :: status (s_invalid);
        else if (val [0] == '_')
            if ((val != "_blank") && (val != "_self") && (val != "_parent") && (val != "_top"))
                string_value < t_target > :: status (s_invalid);  }
    ::std::string diagnose () const
    {   if (! string_value < t_target > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        if (string_value < t_target > :: get_value ().empty ()) return "must not be empty";
        ::std::string s (quote (get_value ()));
        s += " starts with '_', but is neither _blank, _self, _parent, nor _top";
        return s; } };

template < > struct type_master < t_id > : public string_value < t_id >
{   bool tested_ = false, predefined_ = false;
    void swap (type_master < t_id >& t) noexcept
    {   ::std::swap (tested_, t.tested_);
        ::std::swap (predefined_, t.predefined_);
        type_base < t_id >::swap (t); }
    void set_value (const ::std::string& s)
    {   string_value < t_id > :: set_value (s);
        if (s.empty () || (s.find (' ') != s.npos))
            string_value < t_id > :: status (s_invalid);
        tested_ = predefined_ = false; }
    ::std::string diagnose () const
    {   if (! string_value < t_id > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        ::std::string s (quote (string_value < t_id > :: get_value ()));
        if (tested_ && predefined_) s += " is already defined";
        else s += " must not be empty, may not contain a space, and must be unique";
        return s; }
   bool invalid_id (sstr_t* ids)
   {    assert (ids != nullptr);
        if (bad ()) return false;
        if (! tested_)
        {   ::std::string s ( string_value < t_id > :: get_value ());
            predefined_ = (ids -> find (s) != ids -> end ());
            if (! predefined_) ids -> insert (s);
            else string_value < t_id > :: status (s_invalid);
            tested_ = true; }
        return predefined_; } };

template < > struct type_master < t_illegal > : public type_base < t_illegal >
{   void set_value (const ::std::string& )
    {   type_base < t_illegal > :: status (s_invalid); }
    ::std::string diagnose () const
    {    if (! context.tell (e_error)) return ::std::string ();
         return ::std::string ("always illegal"); } };

template < > struct type_master < t_is > : public string_value < t_is >
{   void set_value (const ::std::string& s)
    {   string_value < t_is > :: set_value (s);
        if (s.empty ()) string_value < t_is > :: status (s_invalid); }
    ::std::string diagnose () const
    {   if (! string_value < t_is > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        ::std::string s (quote (string_value < t_is > :: get_value ()));
        s += " is not the name of a custom element";
        return s; } };

template < > struct type_master < t_key > : public string_value < t_key >
{   void set_value (const ::std::string& s)
    {   string_value < t_key > :: set_value (s);
        if ((s.length () != 1) || (s [0] <= ' '))
            string_value < t_key > :: status (s_invalid); }
    ::std::string diagnose () const
    {   if (! string_value < t_key > ::invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        return "a single character is expected, which cannot be a space"; } };

template < > struct type_master < t_integer > : public numeric_value < t_integer, int >
{   ::std::string diagnose () const
    {   if (! numeric_value < t_integer, int > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        ::std::string s (quote (numeric_value < t_integer, int > :: get_value ()));
        s += " is not an integer.";
        return s; } };

template < > struct type_master < t_lang > : public string_value < t_lang > // should check values
{   void set_value (const ::std::string& s) // sanity test only
    {   string_value < t_lang > :: set_value (s);
        if ((s.length () < 2) || ((s.length () > 3) && (s.find ('-') == s.npos)))
            string_value < t_lang > :: status (s_invalid); }
    ::std::string diagnose () const
    {   if (! string_value < t_lang > ::invalid ()) return ::std::string ();
        if (! context.tell (e_info)) return ::std::string ();
        ::std::string res ("'");
        res += quote (string_value < t_lang > :: get_value ());
        res += "' is an unknown language code (see https://tools.ietf.org/html/bcp47)";
        return res; } };

template < > struct type_master < t_mime > : public string_value < t_mime >   // should check values
{   void set_value (const ::std::string& s) // sanity test only
    {   string_value < t_mime > :: set_value (s);
        if (s.find ('/') == s.npos)
            string_value < t_mime > :: status (s_invalid); }
    ::std::string diagnose () const
    {   if (! string_value < t_mime > :: invalid ()) return ::std::string ();
        if (! context.tell (e_info)) return ::std::string ();
        ::std::string res ("'");
        res += quote (string_value < t_mime > ::get_value ());
        res += "' does not appear to be a standard mime type";
        return res; } };

template < > struct type_master < t_rating > : public numeric_value < t_rating, int >
{   void set_value (const ::std::string& s)
    {   numeric_value < t_rating, int > :: set_value (s);
        if ((s.length () != 1) || (s [0] < '1') || (s [0]> '5'))
            numeric_value < t_rating, int > :: status (s_invalid); }
    ::std::string diagnose () const
    {   if (! numeric_value < t_rating, int > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        return "a digit between 1 and 5 inclusive is expected"; } };

template < > struct type_master < t_real > : public type_base < t_real >
{   double value_ = 0.0;
    void swap (type_master < t_real >& t) noexcept
    {   ::std::swap (value_, t.value_);
        type_base < t_real >::swap (t); }
    ::std::string get_value () const { return ::boost::lexical_cast < ::std::string > (value_); }
    void set_value (const ::std::string& s)
    {   value_ = lexical < double > :: cast (trim_the_lot_off (s), 0.0);
        if (lexical < double > :: test (s)) type_base < t_real > :: status (s_good);
        else type_base < t_real > :: status (s_invalid); }
    ::std::string diagnose () const
    {   if (! type_base < t_real > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        ::std::string s (quote (get_value ()));
        s += " is not a floating point number";
        return s; } };

template < > struct type_master < t_sex > : public string_value < t_sex >
{   char sex_ = 0;
    ::std::string gender_;
    void swap (type_master < t_sex >& t) noexcept
    {   ::std::swap (sex_, t.sex_);
        gender_.swap (t.gender_);
        type_base < t_sex >::swap (t); }
    bool parse (const ::std::string& s)
    {  size_t pos = s.find (';');
        switch (pos)
        {   case 0 :
                sex_ = 0;
                gender_ = s.substr (1);
                return true;
            case 1 :
                sex_ = s [0];
                gender_ = s.substr (2);
                break;
            case ::std::string::npos :
                if (s.length () != 1) return false;
                sex_ = s [0];
                break;
            default:
                return false; }
        return (s.substr (0, 1).find_first_of ("FfMmNnOoUu") != s.npos); }
    void set_value (const ::std::string& s)
    {   string_value < t_sex > :: set_value (s);
        if (! parse (s)) string_value < t_sex > :: status (s_invalid); }
    char sex () const { return sex_; }
    ::std::string gender () const { return gender_; }
    ::std::string diagnose () const
    {   if (! string_value < t_sex > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        ::std::string res (quote (string_value < t_sex > ::get_value ()));
        res += " should be one of F, M, N, O, U, or otherwise conform to RFC 6350";
        return res; } };

template < > struct type_master < t_sizes > : public type_base < t_sizes >
{   bool any_ = false;
    int x_ = 0, y_ = 0;
    void swap (type_master < t_sizes >& t)
    {   ::std::swap (any_, t.any_);
        ::std::swap (x_, t.x_);
        ::std::swap (y_, t.y_);
        type_base <t_sizes >::swap (t); }
    bool parse (const ::std::string& s)
    {   any_ = (s == "any");
        if (any_) return true;
        size_t sep = s.find (',');
        if (sep == s.npos) return false;
        bool b = false;
        x_ = lexical < int > :: cast2 (s.substr (0, sep), b);
        if (b) y_ = lexical < int > :: cast2 (s.substr (sep + 1), b);
        return b; }
    ::std::string get_value () const
    {   if (any_) return "any";
        ::std::string s (::boost::lexical_cast < ::std::string > (x_));
        s += ",";
        s = ::boost::lexical_cast < ::std::string > (y_);
        return s; }
    void set_value (const ::std::string& s)
    {   if (parse (s)) type_base < t_sizes > :: status (s_good);
        else type_base < t_sizes > :: status (s_invalid); }
    void reset () { any_ = false; x_ = y_ = 0; type_base < t_sizes > :: reset (); }
    ::std::string diagnose () const
    {   if (! type_base < t_sizes > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        ::std::string s (quote (type_base < t_sizes > :: get_value ()));
        s += " should be 'any', or a pair of (comma separated) integers";
        return s; } };

template < > struct type_master < t_srcset > : public string_value < t_srcset >
{   void set_value (const ::std::string& s) // sanity test only
    {   string_value < t_srcset > :: set_value (s);
        if (s.empty ()) string_value < t_srcset > :: status (s_invalid); }
    ::std::string diagnose () const
    {   if (! string_value < t_srcset > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        return "a srcset cannot be empty"; } };

template < > struct type_master < t_unsigned > : public numeric_value < t_unsigned, unsigned int >
{   ::std::string diagnose () const
    {   if (! numeric_value < t_unsigned, unsigned int > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        return "unsigned integer expected"; } };

template < > struct type_master < t_useid > : public string_value < t_useid >
{   void set_value (const ::std::string& s)
    {   string_value < t_useid > :: set_value (s);
        if (s.empty () || s.find (' ') != s.npos)
            string_value < t_useid > :: status (s_invalid); }
    bool invalid_id (sstr_t* ids)
    {   assert (ids != nullptr);
        if (ids -> find (string_value < t_useid > :: get_value ()) != ids -> end ()) return false;
        string_value < t_useid > :: status (s_invalid);
        return true; }
    ::std::string diagnose () const
    {   if (! string_value < t_useid > :: invalid ()) return ::std::string ();
        if (! context.tell (e_error)) return ::std::string ();
        ::std::string s (string_value < t_useid > :: get_value ());
        s += " must refer to an existing identifier";
        return s; } };

// investigate product codes for ISBN, etc.. http://microformats.org/wiki/h-product
