/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence,  or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#pragma once

#include "type_master.h"

class directory;
class url;

bool verify_directory_url (const directory& d, const ::std::string& url, ::std::ostringstream& ss);
bool url_sanity_test (const url& u);
::std::string url_verify_id (const sstr_t* ssi, const ::std::string& original, const ::std::string& fragment);

template < > struct type_master < t_url > : public type_base < t_url >  // should check values
{   url value_;
    ::std::string diagnosis_;
    ::std::string get_value () const { return value_.original (); }
    static bool is_url () { return true; }
    void set_value (const ::std::string& s)
    {   value_ = s;
        diagnosis_ += value_.diagnosis ();
        type_base < t_url > :: status (s_good); }
    void swap (type_master < t_url >& t) noexcept
    {   value_.swap (t.value_);
        diagnosis_.swap (t.diagnosis_);
        type_base < t_url >::swap (t); }
    void reset ()
    {   value_.reset ();
        diagnosis_.clear ();
        type_base < t_url > :: reset (); }
    bool verify_url (const directory& d)
    {   if (! context.links ()) return true;
        ::std::ostringstream ss;
        bool res = verify_directory_url (d, get_value (), ss);
        diagnosis_ += ss.str ();
        return res; }
    void verify_id (sstr_t* ssi)
    {   append (diagnosis_, "; ", url_verify_id (ssi, value_.get_component (es_original), value_.get_component (es_fragment))); }
    ::std::string original () const { return get_value (); }
    ::std::string diagnose () const
    {   if (! context.tell (e_error)) return ::std::string ();
        if (! value_.is_local () && ! context.tell (e_warning)) return ::std::string ();
        return diagnosis_; } };

template < > struct type_master < t_schema > : public type_master < t_url >
{   bool verify_url (const directory& )
    {   return true; }  // in future verify against standard schemas
    static bool is_url () { return false; }
    ::std::string diagnose () const
    {   if (! context.tell (e_warning)) return ::std::string ();
        return diagnosis_; } };

template < > struct type_master < t_urls > : public string_value < t_urls >
{   vurl_t value_;
    ::std::string diagnosis_;
    ::std::string get_value () const
    {   ::std::string s;
        for (auto& u : value_)
        {   if (! s.empty ()) s += ",";
            s += u.original (); }
        return s; }
    void set_value (const ::std::string& s)
    {   value_ = split_urls_by_space (s);
        for (auto& u : value_)
            if (u.invalid ())
            {   type_base < t_urls > :: status (s_invalid);
                return; }
        type_base < t_urls > :: status (s_good); }
    void swap (type_master < t_urls >& t) noexcept
    {   value_.swap (t.value_);
        diagnosis_.swap (t.diagnosis_);
        type_base < t_urls > :: swap (t); }
    void reset ()
    {   value_.clear ();
        diagnosis_.clear ();
        type_base < t_urls > :: reset (); }
    static bool is_url () { return true; }
    bool verify_url (const directory& d)
    {   if (! context.links ()) return true;
        ::std::ostringstream ss;
        bool res = true;
        for (auto& u : value_)
            if (! verify_directory_url (d, u.original (), ss))
                res = false;
        diagnosis_ = ss.str ();
        return res; }
    void verify_id (sstr_t* ssi)
    {   ::std::string res;
        assert (ssi != nullptr);
        for (auto& u : value_)
            append (res, ", ", url_verify_id (ssi, u.get_component (es_original), u.get_component (es_fragment)));
        append (diagnosis_, "; ", res); }
    ::std::string diagnose () const
    {   if (! context.tell (e_error)) return ::std::string ();
        return diagnosis_; } };

// add checks for t_identifier_url (see h-product) plus url schemes